/*   Copyright Anritsu 2002  All Rights Reserved.                            */
/*****************************************************************************/

#include <formatio.h>
#include <visa.h>
#include <vpptype.h>
#include <ansi_c.h>
#include <string.h>
#include <analysis.h>
#include "MT8820.h"


#define AUMT8820_GSM_REVISION           "Rev 3.1,01Sep2003, CVI 5.5"     /* Instrument driver revision */


/* = MT8820 Radio Communication Analyzer (VISA) ======= */
/* 
    LabWindows 5.5 Instrument Driver
    Original Release: September 2002
    Modification History:

        Sep 2002 -  Original version
        12 Oct 2002 -  Check in Japan
        10 Jan 2003 -  Added fp files
        25 Jul 2003 -  [V3.0]  New Released
        01 Sep 2003 -  [V3.1]  Anritsu MCA1 
        					   Added:ALLMEAS? GPRSBER
        					   Modified:BER?/BER_SAMPLE[?]/LBTYPE[?]
        
        

*/
/* ========================================================================= */

/*****************************************************************************/
/*								     GPRS	 								 
			    The following functions are defined for GPRS				 */
/*****************************************************************************/

/*****************************************************************************/
/*
       Function:   Set Operating Mode
       Command(s): OPEMODE[?]
       Purpose:    Get/Set Operating Mode
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Operating_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"GSM", "GPRS", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "OPEMODE %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Operating Mode
       Command(s): OPEMODE[?]
       Purpose:    Get/Set Operating Mode
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Operating_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"GSM", "GPRS", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OPEMODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set External Down Link Loss
       Command(s): DLEXTLOSS[?]
       Purpose:    Get/Set External Downlink loss level in each band
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_External_Down_Link_Loss(ViSession instrSession, ViInt32 band, ViReal64 loss)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"BAND1", "BAND2", "BAND3", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(loss,-55.0,55.0) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
      
   len = sprintf(buf, "DLEXTLOSS %s,%f", commandArray[band], loss);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get External Down Link Loss
       Command(s): DLEXTLOSS[?]
       Purpose:    Get/Set External Downlink loss level in each band
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_External_Down_Link_Loss(ViSession instrSession, ViInt32 band, ViReal64 _VI_FAR *loss)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"BAND1", "BAND2", "BAND3", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DLEXTLOSS? %s", commandArray[band]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", loss);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set External Up Link Loss
       Command(s): ULEXTLOSS[?]
       Purpose:    Get/Set External Uplink loss level in each band
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_External_Up_Link_Loss(ViSession instrSession, ViInt32 band, ViReal64 loss)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"BAND1", "BAND2", "BAND3", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(loss,-55.0,55.0) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
      
   len = sprintf(buf, "ULEXTLOSS %s,%f", commandArray[band], loss);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get External Up Link Loss
       Command(s): ULEXTLOSS[?]
       Purpose:    Get/Set External Uplink loss level in each band
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_External_Up_Link_Loss(ViSession instrSession, ViInt32 band, ViReal64 _VI_FAR *loss)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"BAND1", "BAND2", "BAND3", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ULEXTLOSS? %s", commandArray[band]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", loss);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set External Aux Loss
       Command(s): AUEXTLOSS[?]
       Purpose:    Get/Set External Auxillary loss level in each band
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_External_Aux_Loss(ViSession instrSession, ViInt32 band, ViReal64 loss)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"BAND1", "BAND2", "BAND3", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(loss,-55.0,55.0) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
      
   len = sprintf(buf, "AUEXTLOSS %s,%f", commandArray[band], loss);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get External Aux Loss
       Command(s): AUEXTLOSS[?]
       Purpose:    Get/Set External Auxillary loss level in each band
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_External_Aux_Loss(ViSession instrSession, ViInt32 band, ViReal64 _VI_FAR *loss)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"BAND1", "BAND2", "BAND3", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AUEXTLOSS? %s", commandArray[band]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", loss);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set External Loss Mode
       Command(s): EXTLOSSW[?]
       Purpose:    Get/Set External Loss Mode
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_External_Loss_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "EXTLOSSW %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get External Loss Mode
       Command(s): EXTLOSSW[?]
       Purpose:    Get/Set External Loss Mode
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_External_Loss_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "EXTLOSSW?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Output Level Correction
       Command(s): OLCOR[?]
       Purpose:    Get/Set Output Level Correction value
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Output_Level_Correction(ViSession instrSession, ViInt32 channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"CCH", "TCH", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "OLCOR %s", commandArray[channel]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Output Level Correction
       Command(s): OLCOR[?]
       Purpose:    Get/Set Output Level Correction value
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Output_Level_Correction(ViSession instrSession, ViInt32  _VI_FAR *channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"CCH", "TCH", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OLCOR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, channel)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Connection Type
       Command(s): CONNTYPE[?]
       Purpose:    Get/Set Connection Type
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Connection_Type(ViSession instrSession, ViInt32 type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MODEA", "MODEB", "BLER", "EXT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CONNTYPE %s", commandArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Connection Type
       Command(s): CONNTYPE[?]
       Purpose:    Get/Set Connection Type
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Connection_Type(ViSession instrSession, ViInt32  _VI_FAR *type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MODEA", "MODEB","BLER", "EXT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CONNTYPE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, type)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Coding Scheme
       Command(s): CS[?]
       Purpose:    Get/Set The coding scheme
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Coding_Scheme(ViSession instrSession, ViInt32 scheme)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"CS1", "CS2", "CS3", "CS4", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CS %s", commandArray[scheme]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Coding Scheme
       Command(s): CS[?]
       Purpose:    Get/Set The coding scheme
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Coding_Scheme(ViSession instrSession, ViInt32  _VI_FAR *scheme)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"CS1", "CS2", "CS3", "CS4", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, scheme)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Net Bit Rate
       Command(s): NBRATE[?]
       Purpose:    Get/Set Net Bit Rate
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Net_Bit_Rate(ViSession instrSession, ViInt32 rate)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"13.0KBPS", "12.2KBPS", "11.4KBPS", "10.2KBPS", "7.95KBPS", "7.40KBPS", "6.70KBPS", "5.90KBPS", "5.15KBPS","4.75KBPS", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "NBRATE %s", commandArray[rate]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Net Bit Rate
       Command(s): NBRATE[?]
       Purpose:    Get/Set Net Bit Rate
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Net_Bit_Rate(ViSession instrSession, ViInt32  _VI_FAR *rate)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"13.0KBPS", "12.2KBPS", "11.4KBPS", "10.2KBPS", "7.95KBPS", "7.40KBPS", "6.70KBPS", "5.90KBPS", "5.15KBPS","4.75KBPS", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "NBRATE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, rate)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Multi Slot Configuration
       Command(s): MLTSLTCFG[?]
       Purpose:    Get/Set Multi Slot Configuration
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Multi_Slot_Configuration(ViSession instrSession, ViInt32 configuration)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"1DL1UL", "2DL1UL", "2DL2UL", "3DL1UL", "3DL2UL", "4DL1UL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "MLTSLTCFG %s", commandArray[configuration]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Multi Slot Configuration
       Command(s): MLTSLTCFG[?]
       Purpose:    Get/Set Multi Slot Configuration
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Multi_Slot_Configuration(ViSession instrSession, ViInt32  _VI_FAR *configuration)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"1DL1UL", "2DL1UL", "2DL2UL", "3DL1UL", "3DL2UL", "4DL1UL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MLTSLTCFG?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, configuration)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Location ID RAC
       Command(s): RAC[?]
       Purpose:    Get/Set Location ID RAC
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Location_ID_RAC(ViSession instrSession, ViInt32 slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(slot,0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "RAC %d", slot);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Location ID RAC
       Command(s): RAC[?]
       Purpose:    Get/Set Location ID RAC
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Location_ID_RAC(ViSession instrSession,  ViInt32 _VI_FAR *slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RAC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", slot);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Network Control Mode
       Command(s): NWCTRLMODE[?]
       Purpose:    Get/Set Network Control Mode
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Network_Control_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NC0", "NC1", "NC2", "RESET", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "NWCTRLMODE %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Network Control Mode
       Command(s): NWCTRLMODE[?]
       Purpose:    Get/Set Network Control Mode
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Network_Control_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NC0", "NC1", "NC2", "RESET", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "NWCTRLMODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Block Error Rate Polling Interval
       Command(s): BLERPOLINT[?]
       Purpose:    Get/Set Block Error Rate Polling Interval
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Block_Error_Rate_Polling_Interval(ViSession instrSession, ViInt32 interval)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(interval,1,32) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "BLERPOLINT %d", interval);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Polling Interval
       Command(s): BLERPOLINT[?]
       Purpose:    Get/Set Block Error Rate Polling Interval
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Polling_Interval(ViSession instrSession,  ViInt32 _VI_FAR *interval)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLERPOLINT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", interval);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Block Error Rate Number Of Samples
       Command(s): BLER_SAMPLE[?]
       Purpose:    Get/Set Block Error Rate Number of Samples
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Block_Error_Rate_Number_Of_Samples(ViSession instrSession, ViInt32 samples)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(samples,0,99999999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "BLER_SAMPLE %d", samples);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Number Of Samples
       Command(s): BLER_SAMPLE[?]
       Purpose:    Get/Set Block Error Rate Number of Samples
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Number_Of_Samples(ViSession instrSession,  ViInt32 _VI_FAR *samples)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLER_SAMPLE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", samples);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Block Error Rate Measurement Mode
       Command(s): BLER_MEAS[?]
       Purpose:    Get/Set Block Error Rate Measurementl Mode
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Block_Error_Rate_Measurement_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BLER_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Measurement Mode
       Command(s): BLER_MEAS[?]
       Purpose:    Get/Set Block Error Rate Measurementl Mode
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Measurement_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLER_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Limit Setting Window
       Command(s): VWOPN[?]
       Purpose:    Get/Set Limit Setting Window
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Limit_Setting_Window(ViSession instrSession, ViInt32 open_Window)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"LMSET", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "VWOPN %s", commandArray[open_Window]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Limit Setting Window
       Command(s): VWOPN[?]
       Purpose:    Get/Set Limit Setting Window
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Limit_Setting_Window(ViSession instrSession, ViInt32  _VI_FAR *open_Window)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"LMSET", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "VWOPN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, open_Window)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate
       Command(s): BLER?
       Purpose:    Get Block Error Rate
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate(ViSession instrSession, ViReal64 _VI_FAR *ratio)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLER?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", ratio);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Count
       Command(s): BLERCNT?
       Purpose:    Get Block Error Rate Count
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Count(ViSession instrSession, ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLERCNT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Receive Count
       Command(s): BLERRECEIVE?
       Purpose:    Get Block Error Rate Receive Count
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Receive_Count(ViSession instrSession, ViReal64 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLERRECEIVE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Total
       Command(s): TTL_BLER?
       Purpose:    Get Block Error Rate Total Measurement Results
       Judgement defined by:  AUMT8820_BLOCK_ERROR_RATE_TOTAL_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Total(ViSession instrSession, ViInt32 _VI_FAR *judgement, ViReal64 _VI_FAR *ratio, ViInt32 _VI_FAR *error_count, ViInt32 _VI_FAR *received_block, ViInt32 _VI_FAR *samples)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_BLER?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%d,%d,%d", judgement, ratio, error_count, received_block, samples);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get C Value Query
       Command(s): CALLCVALUE?
       Purpose:    Get C Value Results
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_C_Value_Query(ViSession instrSession, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLCVALUE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d", flag, result);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Signal Variance Query
       Command(s): CALLSIGVAR?
       Purpose:    Get Signal Variance Results
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Signal_Variance_Query(ViSession instrSession, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLSIGVAR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d", flag, result);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Interference Level Query
       Command(s): CALLITFLVL?
       Purpose:    Get Interference Level Results
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Interference_Level_Query(ViSession instrSession, ViInt32 slot, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(slot,0,7) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
      
   len = sprintf(buf, "CALLITFLVL? %d", slot);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d", flag, result);

   return auMT8820_status;

}

/*****************************************************************************/
/*								     GSM	 								 
			    The following functions are defined for GSM					 */
/*****************************************************************************/

/*****************************************************************************/
/*
       Function:   Set Audio Input Level
       Command(s): AIFLSCL[?]
       Purpose:    Set/Get the audio input full scale level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Audio_Input_Level(ViSession instrSession, ViReal64 level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(level, 1.000000, 5000.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   if (level <= 50.00)
     len = sprintf(buf, "AIFLSCL %.2lf", level);
   else
    {
     if (level <= 500.0)
       len = sprintf(buf, "AIFLSCL %.1lf", level);
     else
        len = sprintf(buf, "AIFLSCL %.0lf", level);
    }
     
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Audio Input Level
       Command(s): AIFLSCL[?]
       Purpose:    Set/Get the audio input full scale level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Audio_Input_Level(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AIFLSCL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Encoder Input Gain
       Command(s): AIGAIN[?]
       Purpose:    Set/Get the encoder input gain (dB)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Encoder_Input_Gain(ViSession instrSession, ViReal64 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "AIGAIN %.2lf", level_dB);

   if (AuMT8820_InvalidViReal64Range(level_dB, -3.000000, 3.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Encoder Input Gain
       Command(s): AIGAIN[?]
       Purpose:    Set/Get the encoder input gain
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Encoder_Input_Gain(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AIGAIN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Audio Input Output Connector 
       Command(s): AINOUT[?]
       Purpose:    Set/Get the audio input/output connector
       Connector defined by:  AUMT8820_AUDIO_INPUT_OUTPUT_CONNECTOR__xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Audio_Input_Output_Connector(ViSession instrSession, ViInt32 connector)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"HANDSET", "AF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "AINOUT %s", commandArray[connector]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Audio Input Output Connector 
       Command(s): AINOUT[?]
       Purpose:    Set/Get the audio input/output connector
       Connector defined by:  AUMT8820_AUDIO_INPUT_OUTPUT_CONNECTOR__xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Audio_Input_Output_Connector(ViSession instrSession, ViInt32  _VI_FAR *connector)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"HANDSET", "AF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AINOUT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, connector)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get All Fundamental Measurement Results
       Command(s): ALLMEAS?
       Purpose:    Query all fundamental measurement results
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_All_Fundamental_Measurement_Results(ViSession instrSession, ViChar _VI_FAR meas_Results[]) 
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s",meas_Results);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Meas Fundamental Measurement Results
       Command(s): ALLMEAS? PWR
       Purpose:    Query all fundamental measurement results
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Meas_Fundamental_Measurement_Results(ViSession instrSession, ViChar _VI_FAR meas_Results[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS? PWR");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s",meas_Results);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Fundamental Measurement Results
       Command(s): ALLMEAS? VSTIME
       Purpose:    Query all fundamental measurement results
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Fundamental_Measurement_Results(ViSession instrSession, ViChar _VI_FAR meas_Results[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];       
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS? VSTIME");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;
														
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)   
      return auMT8820_status;

   Scan(rdBuf, "%s",meas_Results);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Template Fundamental Measurement Results
       Command(s): ALLMEAS? TEMP
       Purpose:    Query all fundamental measurement results
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Template_Fundamental_Measurement_Results(ViSession instrSession, ViChar _VI_FAR meas_Results[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS? TEMP");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s",meas_Results);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Mod Analysis Fundamental Measurement Results
       Command(s): ALLMEAS? MOD
       Purpose:    Query all fundamental measurement results
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Mod_Analysis_Fundamental_Measurement_Results(ViSession instrSession, ViChar _VI_FAR meas_Results[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS? MOD");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s",meas_Results);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Out Spectrum Mod Fundamental Measurement Results
       Command(s): ALLMEAS? ORFSMD
       Purpose:    Query all fundamental measurement results
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Out_Spectrum_Mod_Fundamental_Measurement_Results(ViSession instrSession, ViChar _VI_FAR meas_Results[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS? ORFSMD");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s",meas_Results);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Out Spectrum Switching Fundamental Measurement Results
       Command(s): ALLMEAS? ORFS
       Purpose:    Query all fundamental measurement results
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Out_Spectrum_Switching_Fundamental_Measurement_Results(ViSession instrSession, ViChar _VI_FAR meas_Results[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS? ORFSSW");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s",meas_Results);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Fundamental Measurement Results
       Command(s): ALLMEAS? BER
       Purpose:    Query all fundamental measurement results
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Fundamental_Measurement_Results(ViSession instrSession, ViChar _VI_FAR meas_Results[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS? BER");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s",meas_Results);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Fundamental Measurement Results
       Command(s): ALLMEAS? BLER
       Purpose:    Query all fundamental measurement results
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Fundamental_Measurement_Results(ViSession instrSession, ViChar _VI_FAR *meas_Results)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS? BLER");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s",meas_Results);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get GPRS Bit Error Rate Fundamental Measurement Results
       Command(s): ALLMEAS? GPRSBER
       Purpose:    Query all fundamental measurement results
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_GPRS_Bit_Error_Rate_Fundamental_Measurement_Results(ViSession instrSession, ViChar _VI_FAR meas_Results[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS? GPRSBER");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s",meas_Results);

   return auMT8820_status;

}



/*****************************************************************************/
/*
       Function:   Set All Fundamental Measurement Items
       Command(s): ALLMEASITEMS
       Purpose:    Batch setting of fundamental measurement items
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_All_Fundamental_Measurement_Items(ViSession instrSession, ViChar _VI_FAR items[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "ALLMEASITEMS %s", items);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get All Fundamental Measurement Items
       Command(s): ALLMEASITEMS?
       Purpose:    Batch reading of fundamental measurement items modes
/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_All_Fundamental_Measurement_Items(ViSession instrSession, ViChar _VI_FAR items[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEASITEMS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s" , items);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Handset Microphone Volume
       Command(s): AMICVOL[?]
       Purpose:    Set/Get handset microphone volume
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Handset_Microphone_Volume(ViSession instrSession, ViInt32 volume)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(volume,0,5) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "AMICVOL %d", volume);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Handset Microphone Volume
       Command(s): AMICVOL[?]
       Purpose:    Set/Get handset microphone volume
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Handset_Microphone_Volume(ViSession instrSession,  ViInt32 _VI_FAR *volume)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AMICVOL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", volume);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Audio Output Level
       Command(s): AOFLSCL[?]
       Purpose:    Set/Get the voltage for decoder full scale output
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Audio_Output_Level(ViSession instrSession, ViReal64 level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(level, 0.000000, 5000.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   if (level <= 50.00)
     len = sprintf(buf, "AOFLSCL %.2lf", level);

   else
    {
     if (level <= 500.0)
       len = sprintf(buf, "AOFLSCL %.1lf", level);
     else
        len = sprintf(buf, "AOFLSCL %.0lf", level);
	}
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Audio Output Level
       Command(s): AOFLSCL[?]
       Purpose:    Set/Get the voltage for decoder full scale output
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Audio_Output_Level(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AOFLSCL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Handset Speaker Volume
       Command(s): ASPVOL[?]
       Purpose:    Set/Get handset speaker output level (volume)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Handset_Speaker_Volume(ViSession instrSession, ViInt32 volume)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(volume,0,5) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "ASPVOL %d", volume);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Handset Speaker Volume
       Command(s): ASPVOL[?]
       Purpose:    Set/Get handset speaker output level (volume)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Handset_Speaker_Volume(ViSession instrSession,  ViInt32 _VI_FAR *volume)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ASPVOL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", volume);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Average
       Command(s): AVG_CARRF?
       Purpose:    Query average value of carrier frequency
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Average(ViSession instrSession, ViReal64 _VI_FAR *frequency_Hz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_CARRF?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", frequency_Hz);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Error Average
       Command(s): AVG_CARRFERR? unit
       Purpose:    Query average value of carrier frequency error
       Unit defined by  :AUMT8820_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Error_Average(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"HZ", "PPM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_CARRFERR? %s",unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", frequency);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Lower Average
       Command(s): AVG_LMODPWR? offset
       Purpose:    Query average value of the measured results for the average count of the output RF spectrum (modulation, lower)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Lower_Average(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *power)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_LMODPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", power);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Lower Average
       Command(s): AVG_LSWPWR? offset,unit
       Purpose:    Query average value among measured results for the average count of the output RF spectrum (switching, lower)
       Unit defined by  :AUMT8820_UNIT3_xxx
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Lower_Average(ViSession instrSession, ViInt32 offset_freq, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "DB", 0L};
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_LSWPWR? %s,%s", offset_freqArray[offset_freq], unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Magnitude Error Average
       Command(s): AVG_MAGTDERR?
       Purpose:    Query the average value of magnitude error measurement results of average count
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Magnitude_Error_Average(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_MAGTDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Flatness MAX Power Average
       Command(s): AVG_MAXPWR?
       Purpose:    Query the average value of difference (unit: dB) between TX Power and burst on interval maximum power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Flatness_MAX_Power_Average(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_MAXPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Flatness MIN Power Average
       Command(s): AVG_MINPWR?
       Purpose:    Query the average value of difference (unit: dB) between TX Power and burst on interval minimum power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Flatness_MIN_Power_Average(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_MINPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Off Power Average
       Command(s): AVG_OFFPWR? unit
       Purpose:    Query the average value of power at carrier off
       Unit defined by  :AUMT8820_UNIT2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Off_Power_Average(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_OFFPWR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RMS Phase Error Average
       Command(s): AVG_PHASEERR?
       Purpose:    Query the average value of phase vector error
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RMS_Phase_Error_Average(ViSession instrSession, ViReal64 _VI_FAR *degrees)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_PHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", degrees);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Phase Error Average
       Command(s): AVG_PPHASEERR?
       Purpose:    Query the average value of peak phase error
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Phase_Error_Average(ViSession instrSession, ViReal64 _VI_FAR *degrees)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_PPHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", degrees);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power vs Time Leading Average
       Command(s): AVG_PTLEAD?
       Purpose:    Query the average value of power vs. time leading
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_vs_Time_Leading_Average(ViSession instrSession, ViInt32 time, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(time, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "AVG_PTLEAD? %d", time);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power vs Time Trailing Average
       Command(s): AVG_PTTRAIL?
       Purpose:    Query the average value of power vs. time trailing
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_vs_Time_Trailing_Average(ViSession instrSession, ViInt32 time, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(time, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "AVG_PTTRAIL? %d", time);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Template Judgment Average
       Command(s): AVG_PWRTEMP?
       Purpose:    Query the judgment results at average value of template
       Template defined by:  AUMT8820_AVERAGE_TEMPLATE_JUDGMENT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Template_Judgment_Average(ViSession instrSession, ViInt32 _VI_FAR *template)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString comandArray[] = {"PASS", "FAIL", "*", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_PWRTEMP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(comandArray, resBuf0, template)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Ratio Average
       Command(s): AVG_RATIO?
       Purpose:    Query the average value of difference (unit: dB) between TX power and carrier off power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Ratio_Average(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_RATIO?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Time Alignment Average
       Command(s): AVG_TMALMENT?
       Purpose:    Query the average value of time alignment
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Time_Alignment_Average(ViSession instrSession, ViReal64 _VI_FAR *bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_TMALMENT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", bit);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TX Power Average
       Command(s): AVG_TXPWR?
       Purpose:    Query the average value of transmission (Tx) power
       Unit defined by  :AUMT8820_UNIT2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TX_Power_Average(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_TXPWR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Upper Average
       Command(s): AVG_UMODPWR?  offset
       Purpose:    Query the average value of output RF spectrum (modulation, upper)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Upper_Average(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_UMODPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Upper Average
       Command(s): AVG_USWPWR? offset,unit
       Purpose:    Query the average value of output RF spectrum (switching, upper)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
       Unit defined by  :AUMT8820_UNIT3_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Upper_Average(ViSession instrSession, ViInt32 offset_freq, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   static ViString unitArray[] = {"DBM", "DB", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_USWPWR? %s,%s", offset_freqArray[offset_freq], unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Band Calibration
       Command(s): BANDCAL
       Purpose:    Performs calibration within W-CDMA frequency bandwidth
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Band_Calibration(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BANDCAL");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Ratio
       Command(s): BER?
       Purpose:    Query the bir error rate ratio
       Type defined by: AUMT8820_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Ratio(ViSession instrSession, ViInt32 type, ViReal64 _VI_FAR *ratio)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString typeArray[] = {"FER", "CIB", "CII", "FAST", "BER", "GPRSBER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER? %s", typeArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", ratio);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Input Data Clock Edge
       Command(s): BER_INPUTCLK[?]
       Purpose:    Set/Get the starting position of the input data clock edge for the bit error rate measurement
       Clock_Edge defined by:  AUMT8820_BIT_ERROR_RATE_INPUT_DATA_CLOCK_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Input_Data_Clock_Edge(ViSession instrSession, ViInt32 clock_Edge)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"RISE", "FALL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BER_INPUTCLK %s", commandArray[clock_Edge]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Input Data Clock Edge
       Command(s): BER_INPUTCLK[?]
       Purpose:    Set/Get the starting position of the input data clock edge for the bit error rate measurement
       Clock_Edge defined by:  AUMT8820_BIT_ERROR_RATE_INPUT_DATA_CLOCK_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Input_Data_Clock_Edge(ViSession instrSession, ViInt32  _VI_FAR *clock_Edge)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"RISE", "FALL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_INPUTCLK?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, clock_Edge)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Input Polarity
       Command(s): BER_INPUTPOL[?]
       Purpose:    Set/Get the measurement data polarity for bit error rate measurement
       Polarity defined by:  AUMT8820_BIT_ERROR_RATE_INPUT_POLARITY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Input_Polarity(ViSession instrSession, ViInt32 polarity)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"POS", "NEG", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BER_INPUTPOL %s", commandArray[polarity]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Input Polarity
       Command(s): BER_INPUTPOL[?]
       Purpose:    Set/Get the measurement data polarity for bit error rate measurement
       Polarity defined by:  AUMT8820_BIT_ERROR_RATE_INPUT_POLARITY_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Input_Polarity(ViSession instrSession, ViInt32  _VI_FAR *polarity)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"POS", "NEG", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_INPUTPOL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, polarity)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Measurement Mode
       Command(s): BER_MEAS[?]
       Purpose:    Set/Get the bit error rate measurement on/off setting
       Mode defined by:  AUMT8820_BIT_ERROR_RATE_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Measurement_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BER_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Measurement Mode
       Command(s): BER_MEAS[?]
       Purpose:    Set/Get the bit error rate measurement on/off setting
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Measurement_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Measurement Input
       Command(s): BER_MEASINPUT[?]
       Purpose:    Set/Get the input setting for bit error rate measurement
       Input defined by:  AUMT8820_BIT_ERROR_RATE_MEASUREMENT_INPUT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Measurement_Input(ViSession instrSession, ViInt32 input)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"RF", "EXT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BER_MEASINPUT %s", commandArray[input]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Measurement Input
       Command(s): BER_MEASINPUT[?]
       Purpose:    Set/Get the input setting for bit error rate measurement
       Input defined by:  AUMT8820_BIT_ERROR_RATE_MEASUREMENT_INPUT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Measurement_Input(ViSession instrSession, ViInt32  _VI_FAR *input)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"RF", "EXT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_MEASINPUT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, input)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Number Of Samples
       Command(s): BER_SAMPLE[?]
       Purpose:    Set/Get the number of samples bits for bit error rate measurement
       Type defined by: AUMT8820_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Number_Of_Samples(ViSession instrSession, ViInt32 type, ViInt32 samples)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString typeArray[] = {"FER", "CIB", "CII", "FAST", "BER", "GPRSBER",0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(samples,0,99999999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "BER_SAMPLE %s,%d", typeArray[type], samples);    
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Number Of Samples
       Command(s): BER_SAMPLE[?]
       Purpose:    Set/Get the number of samples bits for bit error rate measurement
       Type defined by: AUMT8820_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Number_Of_Samples(ViSession instrSession, ViInt32 type,  ViInt32 _VI_FAR *samples) 
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString typeArray[] = {"FER", "CIB", "CII", "FAST", "BER", "GPRSBER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_SAMPLE? %s", typeArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", samples);
  
   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Count
       Command(s): BERCNT?
       Purpose:    Query the number of error bits during bir error rate measurement
       Type defined by: AUMT8820_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Count(ViSession instrSession, ViInt32 type, ViReal64 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString typeArray[] = {"FER", "CIB", "CII", "FAST", "BER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BERCNT? %s", typeArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", count); 

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Received Bits
       Command(s): BERRECEIVE?
       Purpose:    Query the number of received bits during bir error rate measurement
       Type defined by: AUMT8820_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Received_Bits(ViSession instrSession, ViInt32 type, ViReal64 _VI_FAR *bits)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString typeArray[] = {"FER", "CIB", "CII", "FAST", "BER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BERRECEIVE? %s", typeArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", bits);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Offset
       Command(s): BITOFS[?]
       Purpose:    Set/Get the setting bit offset for transmission measurement
       Offset defined by:  AUMT8820_BIT_OFFSET_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Offset(ViSession instrSession, ViInt32 offset)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"0", "0.5", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BITOFS %s", commandArray[offset]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Offset
       Command(s): BITOFS[?]
       Purpose:    Set/Get the setting bit offset for transmission measurement
       Offset defined by:  AUMT8820_BIT_OFFSET_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Offset(ViSession instrSession, ViInt32  _VI_FAR *offset)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"0", "0.5", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BITOFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, offset)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Neighbour Cell Channel
       Command(s): CALLCA[?]
       Purpose:    Set/Get the neighbour cell channel
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Neighbour_Cell_Channel(ViSession instrSession, ViInt32 cell, ViInt32 channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(cell,1,16) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(channel,-1,1023) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "CALLCA %d,%d", cell, channel);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Neighbour Cell Channel
       Command(s): CALLCA[?]
       Purpose:    Set/Get the neighbour cell channel
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Neighbour_Cell_Channel(ViSession instrSession, ViInt32 cell,  ViInt32 _VI_FAR *channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(cell,1,16) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "CALLCA? %d", cell);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", channel);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Call Processing Error
       Command(s): CALLERR?
       Purpose:    Query the call processing error
       Seq_code defined by:  AUMT8820_CALL_PROCESSING_STATUS_xxx
       Err_code: 0=normal,1-255 is error
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Call_Processing_Error(ViSession instrSession, ViInt32 _VI_FAR *seq_code, ViInt32 _VI_FAR *err_code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d", seq_code, err_code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get IMEI Query
       Command(s): CALLIMEI?
       Purpose:    Query the IMEI
       Flag defined by:  AUMT8820_FLAG_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_IMEI_Query(ViSession instrSession, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLIMEI?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d", flag, id);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get IMSI Query
       Command(s): CALLIMSI?
       Purpose:    Query the IMSI
       Flag defined by:  AUMT8820_FLAG_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_IMSI_Query(ViSession instrSession, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLIMSI?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d", flag, id);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Adjacent Cell Monitoring
       Command(s): CALLNCELL?
       Purpose:    Query the adjacent cell monitor data
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Adjacent_Cell_Monitoring(ViSession instrSession, ViInt32 cell_in,
                                                                                ViInt32 _VI_FAR *flag,
                                                                                ViInt32 _VI_FAR *cell,
                                                                                ViInt32 _VI_FAR *arfcn,
                                                                                ViInt32 _VI_FAR *rxlev,
                                                                                ViInt32 _VI_FAR *ncc,
                                                                                ViInt32 _VI_FAR *bcc)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(cell_in,1,16) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "CALLNCELL? %d", cell_in);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d,%d,%d,%d,%d", flag, cell, arfcn, rxlev, ncc, bcc);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Network Phone Number Query
       Command(s): CALLNWNO?
       Purpose:    Query the network phone number
       Flag defined by:  AUMT8820_FLAG_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Network_Phone_Number_Query(ViSession instrSession, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLNWNO?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d", flag, number);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Call Processing Mode
       Command(s): CALLPROC[?]
       Purpose:    Set/Get the call processing function on/off
       Mode defined by:  AUMT8820_CALL_PROCESSING_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Call_Processing_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CALLPROC %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Call Processing Mode
       Command(s): CALLPROC[?]
       Purpose:    Set/Get the call processing function on/off
       Mode defined by:  AUMT8820_CALL_PROCESSING_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Call_Processing_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLPROC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get MS Power Class
       Command(s): CALLPWRCLASS?
       Purpose:    Query the MS power class
       Flag defined by:  AUMT8820_FLAG_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_MS_Power_Class(ViSession instrSession, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *class)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLPWRCLASS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d", flag, class);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get MS Power Level And Timing Advance
       Command(s): CALLPWRTA?
       Purpose:    Query the MS power level and timing class
       Flag defined by:  AUMT8820_FLAG_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_MS_Power_Level_And_Timing_Advance(ViSession instrSession, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *level, ViInt32 _VI_FAR *value)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLPWRTA?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d,%d", flag, level, value);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RX Level And Quality
       Command(s): CALLREP?
       Purpose:    Query the Rx level and quality
       Flag defined by:  AUMT8820_FLAG_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RX_Level_And_Quality(ViSession instrSession, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *level, ViInt32 _VI_FAR *value)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLREP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d,%d", flag, level, value);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Refresh Call Status
       Command(s): CALLRFR
       Purpose:    Refreshing the call processing, forcibly sets the call processing status to Idle regardless of current status
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Refresh_Call_Status(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CALLRFR");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Call Processing Result
       Command(s): CALLRSLT?
       Purpose:    Query the result for call processing specifying a sequence name for individual processing
       Seq_code defined by:  AUMT8820_CALL_PROCESSING_STATUS_xxx
       Flag defined by:  AUMT8820_FLAG2_xxx
       Err_code: 0=normal, 1-255 is error
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Call_Processing_Result(ViSession instrSession, ViInt32 seq_code, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *err_code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLRSLT? %d", seq_code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d", flag, err_code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Call Start
       Command(s): CALLSA
       Purpose:    Make MT8820 call a terminal, connecting a call
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Call_Start(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CALLSA");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Call Stop
       Command(s): CALLSO
       Purpose:    Make MT8820 release a call, network release
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Call_Stop(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CALLSO");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Call Processing Status
       Command(s): CALLSTAT?
       Purpose:    Query the call processing status
       Seq_code defined by:  AUMT8820_CALL_PROCESSING_STATUS_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Call_Processing_Status(ViSession instrSession, ViInt32 _VI_FAR *seq_code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString comandArray[] = {"", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLSTAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", seq_code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency
       Command(s): CARRF?
       Purpose:    Query the carrier frequency
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency(ViSession instrSession, ViReal64 _VI_FAR *frequency_Hz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CARRF?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", frequency_Hz);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Error
       Command(s): CARRFERR?
       Purpose:    Query the carrier frequency error previous value
       Unit defined by:  AUMT8820_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Error(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"HZ", "PPM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CARRFERR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", frequency);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TCH Channel
       Command(s): CHAN[?]
       Purpose:    Set/Get the TCH channel
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TCH_Channel(ViSession instrSession, ViInt32 channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(channel,0,1023) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "CHAN %d", channel);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TCH Channel
       Command(s): CHAN[?]
       Purpose:    Set/Get the TCH channel
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TCH_Channel(ViSession instrSession,  ViInt32 _VI_FAR *channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CHAN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", channel);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Codec
       Command(s): CODEC[?]
       Purpose:    Set/Get the codec
       Type defined by:  AUMT8820_CODEC_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Codec(ViSession instrSession, ViInt32 type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"FS", "EFS", "HS0", "HS1", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CODEC %s", commandArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Codec
       Command(s): CODEC[?]
       Purpose:    Set/Get the codec
       Type defined by:  AUMT8820_CODEC_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Codec(ViSession instrSession, ViInt32  _VI_FAR *type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"FS", "EFS", "HS0", "HS1", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CODEC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, type)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set CCH Channel
       Command(s): CTRLCH[?]
       Purpose:    Set/Get the CCH channel
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_CCH_Channel(ViSession instrSession, ViInt32 channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(channel,0,1023) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "CTRLCH %d", channel);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get CCH Channel
       Command(s): CTRLCH[?]
       Purpose:    Set/Get the CCH channel
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_CCH_Channel(ViSession instrSession,  ViInt32 _VI_FAR *channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CTRLCH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", channel);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set CCH Downlink Frequency
       Command(s): CTRLDLFREQ[?]
       Purpose:    Set/Get the downlink frequency for CCH
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_CCH_Downlink_Frequency(ViSession instrSession, ViReal64 frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CTRLDLFREQ %fMHz", frequency);

   if (AuMT8820_InvalidViReal64Range(frequency, 30.000000, 2700.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get CCH Downlink Frequency
       Command(s): CTRLDLFREQ[?]
       Purpose:    Set/Get the downlink frequency for CCH
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_CCH_Downlink_Frequency(ViSession instrSession, ViReal64 _VI_FAR *frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   double    freqBuf;
   
   len = sprintf(buf, "CTRLDLFREQ?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", &freqBuf);
   
   *frequency=freqBuf/1000000;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set CCH Uplink Frequency
       Command(s): CTRLULFREQ[?]
       Purpose:    Set/Get the uplink frequency for CCH
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_CCH_Uplink_Frequency(ViSession instrSession, ViReal64 frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CTRLULFREQ %fMHz", frequency);

   if (AuMT8820_InvalidViReal64Range(frequency, 30.000000, 2700.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get CCH Uplink Frequency
       Command(s): CTRLULFREQ[?]
       Purpose:    Set/Get the uplink frequency for CCH
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_CCH_Uplink_Frequency(ViSession instrSession, ViReal64 _VI_FAR *frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   double    freqBuf;
   
   len = sprintf(buf, "CTRLULFREQ?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", &freqBuf);
   
   *frequency=freqBuf/1000000;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TCH Downlink Frequency
       Command(s): DLFREQ[?],RFREQ[?],RXFREQ[?]
       Purpose:    Set/Get the downlink frequency for TCH
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TCH_Downlink_Frequency(ViSession instrSession, ViReal64 frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "DLFREQ %fMHz", frequency);

   if (AuMT8820_InvalidViReal64Range(frequency, 30.000000, 2700.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TCH Downlink Frequency
       Command(s): DLFREQ[?],RFREQ[?],RXFREQ[?]
       Purpose:    Set/Get the downlink frequency for TCH
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TCH_Downlink_Frequency(ViSession instrSession, ViReal64 _VI_FAR *frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   double    freqBuf;
   
   len = sprintf(buf, "DLFREQ?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", &freqBuf);

   *frequency=freqBuf/1000000;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Full Calibration
       Command(s): FULLCAL
       Purpose:    Performs a calibration for all bands of MT8820 input/output
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Full_Calibration(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "FULLCAL");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Handover Type
       Command(s): HOTYPE[?]
       Purpose:    Set/Get the codec
       Type defined by:  AUMT8820_HANDOVER_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Handover_Type(ViSession instrSession, ViInt32 type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"INTRA", "INTER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "HOTYPE %s", commandArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Handover Type
       Command(s): HOTYPE[?]
       Purpose:    Set/Get the codec
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Handover_Type(ViSession instrSession, ViInt32  _VI_FAR *type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"INTRA", "INTER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "HOTYPE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, type)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Input Level
       Command(s): ILVL[?],RFLVL,REFLVL
       Purpose:    Set/Get the input level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Input_Level(ViSession instrSession, ViReal64 Level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "ILVL %.1lf", Level);

   if (AuMT8820_InvalidViReal64Range(Level, -30.000000, 40.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Input Level
       Command(s): ILVL[?],RFLVL,REFLVL
       Purpose:    Set/Get the input level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Input_Level(ViSession instrSession, ViReal64 _VI_FAR *Level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ILVL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", Level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Input Level Control
       Command(s): ILVLCTRL[?]
       Purpose:    Set/Get the input level control
       Type defined by:  AUMT8820_INPUT_LEVEL_CONTROL_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Input_Level_Control(ViSession instrSession, ViInt32 type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PCL", "MANUAL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "ILVLCTRL %s", commandArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Input Level Control
       Command(s): ILVLCTRL[?]
       Purpose:    Set/Get the input level control
       Type defined by:  AUMT8820_INPUT_LEVEL_CONTROL_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Input_Level_Control(ViSession instrSession, ViInt32  _VI_FAR *type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PCL", "MANUAL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ILVLCTRL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, type)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Int Mobile Sub Identity
       Command(s): IMSI[?]
       Purpose:    Set/Get the International Mobile Subscriber Identity (IMSI) when paging
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Int_Mobile_Sub_Identity(ViSession instrSession,  ViChar _VI_FAR id[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "IMSI %s", id);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Int Mobile Sub Identity
       Command(s): IMSI[?]
       Purpose:    Set/Get the International Mobile Subscriber Identity (IMSI) when paging
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Int_Mobile_Sub_Identity(ViSession instrSession, ViChar _VI_FAR id[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "IMSI?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s" ,id);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Parameter Items Displayed
       Command(s): ITEMLIST[?]
       Purpose:    Set/Get the parameter items to be displayed
       Tag defined by:  AUMT8820_PARAMETER_ITEMS_DISPLAYED_xxx
       Option defined by: AUMT8820_PARAMETER_ITEMS_OPTION_xxx 
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Parameter_Items_Displayed(ViSession instrSession, ViInt32 tag, ViInt32 option)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"COM", "AUDIO", "CALL", "TXMEAS", "RXMEAS", "FMEAS", 0L};
   static ViString optionArray[] = {"STD", "NONDSPL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "ITEMLIST %s,%s", commandArray[tag], optionArray[option]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Parameter Items Displayed
       Command(s): ITEMLIST[?]
       Purpose:    Set/Get the parameter items to be displayed
       Tag defined by:  AUMT8820_PARAMETER_ITEMS_DISPLAYED_xxx
       Option defined by: AUMT8820_PARAMETER_ITEMS_OPTION_xxx 
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Parameter_Items_Displayed(ViSession instrSession, ViInt32  tag, ViInt32  _VI_FAR *option)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"COM", "AUDIO", "CALL", "TXMEAS", "RXMEAS", "FMEAS", 0L};
   static ViString optionArray[] = {"STD", "NONDSPL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ITEMLIST? %s", commandArray[tag]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(optionArray, resBuf0, option)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Location ID
       Command(s): LAC[?]
       Purpose:    Set/Get the location area identification
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Location_ID(ViSession instrSession, ViInt32 id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(id,0x0,0x0FFFF) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "LAC %x", id);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Location ID
       Command(s): LAC[?]
       Purpose:    Set/Get the location area identification
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Location_ID(ViSession instrSession,  ViInt32 _VI_FAR *id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LAC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%x", id);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Loop Back Type
       Command(s): LBTYPE[?]
       Purpose:    Set/Get the loop back type when loop back is on
       Type defined by:  AUMT8820_LOOP_BACK_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Loop_Back_Type(ViSession instrSession, ViInt32 type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"FE", "NOFE", "FAST", "FASTSPEECH", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "LBTYPE %s", commandArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Loop Back Type
       Command(s): LBTYPE[?]
       Purpose:    Set/Get the loop back type when loop back is on
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Loop_Back_Type(ViSession instrSession, ViInt32  _VI_FAR *type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"FE", "NOFE", "FAST", "FASTSPEECH", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LBTYPE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, type)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Lower
       Command(s): LMODPWR?
       Purpose:    Query the RF output spectrum (modulation, lower)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx

*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Lower(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LMODPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Loop Back Mode
       Command(s): LOOPBACK[?]
       Purpose:    Set/Get the loop back mode on/off
       Mode defined by:  AUMT8820_LOOP_BACK_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Loop_Back_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "LOOPBACK %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Loop Back Mode
       Command(s): LOOPBACK[?]
       Purpose:    Set/Get the loop back mode on/off
       Mode defined by:  AUMT8820_LOOP_BACK_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Loop_Back_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LOOPBACK?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switch Lower
       Command(s): LSWPWR?
       Purpose:    Query the RF output spectrum (switching, lower)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switch_Lower(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LSWPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RF Output Level Mode
       Command(s): LVL[?]
       Purpose:    Set/Get the RF signal output level mode on/off
       Mode defined by:  AUMT8820_RF_OUTPUT_LEVEL_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RF_Output_Level_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "LVL %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Level Mode
       Command(s): LVL[?]
       Purpose:    Set/Get the RF signal output level mode on/off
       Mode defined by:  AUMT8820_RF_OUTPUT_LEVEL_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Level_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LVL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RF Output Level Continuous Mode
       Command(s): LVLCONT[?]
       Purpose:    Set/Get the RF signal output level un-interruptible change mode on/off.  When set to On the output level can be changed without signal interruption if changed < 30dB
       Mode defined by:  AUMT8820_RF_OUTPUT_LEVEL_CONTINUOUS_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RF_Output_Level_Continuous_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "LVLCONT %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Level Continuous Mode
       Command(s): LVLCONT[?]
       Purpose:    Set/Get the RF signal output level un-interruptible change mode on/off.  When set to On the output level can be changed without signal interruption if changed < 30dB
       Mode defined by:  AUMT8820_RF_OUTPUT_LEVEL_CONTINUOUS_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Level_Continuous_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LVLCONT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Magnitude Error
       Command(s): MAGTDERR?
       Purpose:    Query the magnitude error
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Magnitude_Error(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAGTDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Error Maximum
       Command(s): MAX_CARRFERR?
       Purpose:    Query the maximum value among the average count of frequency errror measurements
       Unit defined by:  AUMT8820_UNIT_xxx

*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Error_Maximum(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"HZ", "PPM", 0L};   
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_CARRFERR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", frequency);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Lower Maximum
       Command(s): MAX_LMODPWR?
       Purpose:    Query the maximum value of the measured results for the average count of the RF output spectrum (modulation, lower)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Lower_Maximum(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_LMODPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Lower Maximum
       Command(s): MAX_LSWPWR?
       Purpose:    Query the maximum value of the measured results for the average count of the RF output spectrum (switching, lower)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
       Unit defined by:  AUMT8820_UNIT3_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Lower_Maximum(ViSession instrSession, ViInt32 offset_freq, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   static ViString unitArray[] = {"DBM", "DB", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_LSWPWR? %s,%s", offset_freqArray[offset_freq], unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Magnitude Error Maximum
       Command(s): MAX_MAGTDERR?
       Purpose:    Query the maximum value among the average count of magnitude error measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Magnitude_Error_Maximum(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_MAGTDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Flatness MAX Power Maximum
       Command(s): MAX_MAXPWR?
       Purpose:    Query the maximum value among the measured results for average count of difference between TX power and burst on interval maximum power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Flatness_MAX_Power_Maximum(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_MAXPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Flatness MIN Power Maximum
       Command(s): MAX_MINPWR?
       Purpose:    Query the maximum value among the measured results for average count of difference between TX power and burst on interval minimum power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Flatness_MIN_Power_Maximum(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_MINPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Off Power Maximum
       Command(s): MAX_OFFPWR?
       Purpose:    Query the maximum value among the measured results for average count of  power at carrier off
       Unit defined by:  AUMT8820_UNIT2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Off_Power_Maximum(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_OFFPWR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RMS Phase Error Maximum
       Command(s): MAX_PHASEERR?
       Purpose:    Query the maximum value among the measured results for average count of  phase vector error measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RMS_Phase_Error_Maximum(ViSession instrSession, ViReal64 _VI_FAR *degrees)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_PHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", degrees);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Phase Error Maximum
       Command(s): MAX_PPHASEERR?
       Purpose:    Query the maximum value among the measured results for average count of  peak phase vector error measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Phase_Error_Maximum(ViSession instrSession, ViReal64 _VI_FAR *degrees)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_PPHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", degrees);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Leading Maximum
       Command(s): MAX_PTLEAD?  time
       Purpose:    Query the maximum value among the measured results for average count of  power vs. time (leading)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Leading_Maximum(ViSession instrSession, ViInt32 time, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(time, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "MAX_PTLEAD? %d", time);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Trailing Maximum
       Command(s): MAX_PTTRAIL?
       Purpose:    Query the maximum value among the measured results for average count of  power vs. time (Trailing)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Trailing_Maximum(ViSession instrSession, ViInt32 time, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(time, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "MAX_PTTRAIL? %d", time);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Template Judgement Maximum
       Command(s): MAX_PWRTEMP?
       Purpose:    Query the judgement results at maximum value among measured results for average count of  the template
       Result defined by:  AUMT8820_MAX_TEMPLATE_JUDGMENT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Template_Judgement_Maximum(ViSession instrSession, ViInt32 _VI_FAR *result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PASS", "FAIL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_PWRTEMP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, result)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Ratio Maximum
       Command(s): MAX_RATIO?
       Purpose:    Query the maximum value among measured results for average count of  the difference between Tx power and carrier off power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Ratio_Maximum(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_RATIO?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Time Alignment Maximum
       Command(s): MAX_TMALMENT?
       Purpose:    Query the maximum value among measured results for average count of  the time alignment
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Time_Alignment_Maximum(ViSession instrSession, ViReal64 _VI_FAR *bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_TMALMENT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", bit);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TX Power Maximum
       Command(s): MAX_TXPWR?
       Purpose:    Query the maximum value among average count of  the transmission power measurements for a phone
       Unit defined by:  AUMT8820_UNIT2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TX_Power_Maximum(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_TXPWR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Upper Maximum
       Command(s): MAX_UMODPWR?
       Purpose:    Query the maximum value among measured results for average count of  the RF output spectrum (modulation, upper)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Upper_Maximum(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_UMODPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Upper Maximum
       Command(s): MAX_USWPWR?
       Purpose:    Query the maximum value among measured results for average count of  the RF output spectrum (switching, upper)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
       Unit defined by:  AUMT8820_UNIT3_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Upper_Maximum(ViSession instrSession, ViInt32 offset_freq, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   static ViString unitArray[] = {"DBM", "DB", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_USWPWR? %s,%s", offset_freqArray[offset_freq], unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Flatness Maximum
       Command(s): MAXPWR?
       Purpose:    Query the difference between Tx power and burst on interval maximum power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Flatness_Maximum(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAXPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Location ID MCC
       Command(s): MCC[?]
       Purpose:    Set/Get the MCC for location area identification
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Location_ID_MCC(ViSession instrSession, ViInt32 id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(id,0,999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "MCC %d", id);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Location ID MCC
       Command(s): MCC[?]
       Purpose:    Set/Get the MCC for location area identification
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Location_ID_MCC(ViSession instrSession,  ViInt32 _VI_FAR *id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MCC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", id);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Software Version
       Command(s): MCFV?
       Purpose:    Query the software version used on the MT8820A
       Mode defined by:  AUMT8820_SOFTWARE_OBJECT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Software_Version(ViSession instrSession, ViInt32 mode, ViChar _VI_FAR version[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MCFV? %d", mode);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s %s", version, version);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Fundamental Measurement Mode
       Command(s): MEASMODE[?]
       Purpose:    Set/Get the mode for fundamental measurements.  A view window cannot be displayed when the mode is set to Fast
       Mode defined by:  AUMT8820_FUNDAMENTAL_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Fundamental_Measurement_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NORM", "FAST", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "MEASMODE %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Fundamental Measurement Mode
       Command(s): MEASMODE[?]
       Purpose:    Set/Get the mode for fundamental measurements.  A view window cannot be displayed when the mode is set to Fast
       Mode defined by:  AUMT8820_FUNDAMENTAL_MEASUREMENT_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Fundamental_Measurement_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NORM", "FAST", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MEASMODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Measurement Object
       Command(s): MEASOBJ[?]
       Purpose:    Set/Get the measurement object
       Object defined by:  AUMT8820_MEASUREMENT_OBJECT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Measurement_Object(ViSession instrSession, ViInt32 object)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MSNB", "RACH", "CONT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "MEASOBJ %s", commandArray[object]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Measurement Object
       Command(s): MEASOBJ[?]
       Purpose:    Set/Get the measurement object
       Object defined by:  AUMT8820_MEASUREMENT_OBJECT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Measurement_Object(ViSession instrSession, ViInt32  _VI_FAR *object)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MS-NB", "RACH", "CONT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MEASOBJ?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, object)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Measurement Slot
       Command(s): MEASSLOT[?]
       Purpose:    Set/Get the measurement slot number for Tx measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Measurement_Slot(ViSession instrSession, ViInt32 slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(slot,0,7) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "MEASSLOT %d", slot);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Measurement Slot
       Command(s): MEASSLOT[?]
       Purpose:    Set/Get the measurement slot number for Tx measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Measurement_Slot(ViSession instrSession,  ViInt32 _VI_FAR *slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MEASSLOT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", slot);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Multiframe Paging Period
       Command(s): MFRMS[?]
       Purpose:    Set/Get the multiframe paging period (BS-PA-MFRMS)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Multiframe_Paging_Period(ViSession instrSession, ViInt32 num)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(num,0,7) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "MFRMS %d", num);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Multiframe Paging Period
       Command(s): MFRMS[?]
       Purpose:    Set/Get the multiframe paging period (BS-PA-MFRMS)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Multiframe_Paging_Period(ViSession instrSession,  ViInt32 _VI_FAR *num)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MFRMS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", num);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Error Minimum
       Command(s): MIN_CARRFERR?
       Purpose:    Query the minimum value among the average count of frequency error measurements
       Unit defined by:  AUMT8820_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Error_Minimum(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"HZ", "PPM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_CARRFERR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", frequency);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Lower Minimum
       Command(s): MIN_LMODPWR?
       Purpose:    Query the minimum value of the measured results for the average count of the RF output spectrum (modulation, lower)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Lower_Minimum(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_LMODPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Lower Minimum
       Command(s): MIN_LSWPWR?
       Purpose:    Query the minimum value of the measured results for the average count of the RF output spectrum (switching, lower)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
       Unit defined by:  AUMT8820_UNIT3_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Lower_Minimum(ViSession instrSession, ViInt32 offset_freq, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   static ViString unitArray[] = {"DBM", "DB", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_LSWPWR? %s,%s", offset_freqArray[offset_freq], unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Magnitude Error Minimum
       Command(s): MIN_MAGTDERR?
       Purpose:    Query the minimum value among average count of the magnitude error measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Magnitude_Error_Minimum(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_MAGTDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Flatness MAX Power Minimum
       Command(s): MIN_MAXPWR?
       Purpose:    Query the minimum value among measured results for average count of the difference between Tx power and burst on interval maximum power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Flatness_MAX_Power_Minimum(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_MAXPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Flatness MIN Power Minimum
       Command(s): MIN_MINPWR?
       Purpose:    Query the minimum value among measured results for average count of the difference between Tx power and burst on interval minimum power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Flatness_MIN_Power_Minimum(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_MINPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Off Power Minimum
       Command(s): MIN_OFFPWR?  unit
       Purpose:    Query the minimum value among measured results for average count of power at carrier off
       Unit defined by:  AUMT8820_UNIT2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Off_Power_Minimum(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_OFFPWR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RMS Phase Error Minimum
       Command(s): MIN_PHASEERR?
       Purpose:    Query the minimum value among average count of phase vector error measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RMS_Phase_Error_Minimum(ViSession instrSession, ViReal64 _VI_FAR *degree)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_PHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", degree);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Phase Error Minimum
       Command(s): MIN_PPHASEERR?
       Purpose:    Query the minimum value among average count of peak phase vector error measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Phase_Error_Minimum(ViSession instrSession, ViReal64 _VI_FAR *degree)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_PPHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", degree);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Leading Minimum
       Command(s): MIN_PTLEAD? time
       Purpose:    Query the minimum value among measured results for average count of power vs time (leading)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Leading_Minimum(ViSession instrSession, ViInt32 time, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(time, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MIN_PTLEAD? %d", time);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Trailing Minimum
       Command(s): MIN_PTTRAIL?  time
       Purpose:    Query the minimum value among measured results for average count of power vs time (Trailing)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Trailing_Minimum(ViSession instrSession, ViInt32 time, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(time, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MIN_PTTRAIL? %d", time);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Template Judgement Minimum
       Command(s): MIN_PWRTEMP?
       Purpose:    Query the judgment results at the minimum value among measured results for the average count of the template
       Result defined by:  AUMT8820_MIN_TEMPLATE_JUDGMENT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Template_Judgement_Minimum(ViSession instrSession, ViInt32 _VI_FAR *result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PASS", "FAIL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_PWRTEMP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, result)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Ratio Minimum
       Command(s): MIN_RATIO?
       Purpose:    Query the minimum value among measured results for the average count of the difference between Tx power and carrier off power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Ratio_Minimum(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_RATIO?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Time Alignment Minimum
       Command(s): MIN_TMALMENT?
       Purpose:    Query the minimum value among measured results for the average count of the time alignment
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Time_Alignment_Minimum(ViSession instrSession, ViReal64 _VI_FAR *bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_TMALMENT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", bit);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Tx Power Minimum
       Command(s): MIN_TXPWR? unit
       Purpose:    Query the minimum value among the average count of the transmission power measurements for a phone
       Unit defined by:  AUMT8820_UNIT2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Tx_Power_Minimum(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_TXPWR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Upper Minimum
       Command(s): MIN_UMODPWR?
       Purpose:    Query the minimum value among measured results for the average count of the RF output spectrum (modulation, upper)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Upper_Minimum(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_UMODPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Upper Minimum
       Command(s): MIN_USWPWR?
       Purpose:    Query the minimum value among measured results for the average count of the RF output spectrum (switching, upper)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
       Unit defined by:  AUMT8820_UNIT3_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Upper_Minimum(ViSession instrSession, ViInt32 offset_freq, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   static ViString unitArray[] = {"DBM", "DB", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_USWPWR? %s,%s", offset_freqArray[offset_freq], unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Flatness Minimum
       Command(s): MINPWR?
       Purpose:    Query the difference between Tx power and burst on interval minimum power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Flatness_Minimum(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MINPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Location ID MNC
       Command(s): MNC[?]
       Purpose:    Set/Get the MNC for location are identification
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Location_ID_MNC(ViSession instrSession, ViInt32 id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(id,0,99) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "MNC %02d", id);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Location ID MNC
       Command(s): MNC[?]
       Purpose:    Set/Get the MNC for location are identification
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Location_ID_MNC(ViSession instrSession,  ViInt32 _VI_FAR *id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MNC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", id);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Modulation Analysis Count
       Command(s): MOD_COUNT[?]
       Purpose:    Set/Get the measurement count for modulation analysis measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Modulation_Analysis_Count(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "MOD_COUNT %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Modulation Analysis Count
       Command(s): MOD_COUNT[?]
       Purpose:    Set/Get the measurement count for modulation analysis measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Modulation_Analysis_Count(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MOD_COUNT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Modulation Analysis Mode
       Command(s): MOD_MEAS[?]
       Purpose:    Set/Get whether to perform the modulation analysis measurement
       Mode defined by:  AUMT8820_MODULATION_ANALYSIS_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Modulation_Analysis_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "MOD_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Modulation Analysis Mode
       Command(s): MOD_MEAS[?]
       Purpose:    Set/Get whether to perform the modulation analysis measurement
       Mode defined by:  AUMT8820_MODULATION_ANALYSIS_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Modulation_Analysis_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MOD_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set MS Power Level
       Command(s): MSPWR[?]
       Purpose:    Set/Get the MS power level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_MS_Power_Level(ViSession instrSession, ViInt32 level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(level,0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "MSPWR %d", level);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get MS Power Level
       Command(s): MSPWR[?]
       Purpose:    Set/Get the MS power level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_MS_Power_Level(ViSession instrSession,  ViInt32 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MSPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Measurement Status
       Command(s): MSTAT?
       Purpose:    Get the current measurement status
       Status defined by: AUMT8820_MEASUREMENT_STATUS_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Measurement_Status(ViSession instrSession, ViInt32 _VI_FAR *status)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MSTAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", status);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Network ID BCC
       Command(s): NIDBCC[?]
       Purpose:    Set/Get the BCC network Identification
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Network_ID_BCC(ViSession instrSession, ViInt32 id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(id,0,7) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "NIDBCC %d", id);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Network ID BCC
       Command(s): NIDBCC[?]
       Purpose:    Set/Get the BCC network Identification
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Network_ID_BCC(ViSession instrSession,  ViInt32 _VI_FAR *id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "NIDBCC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", id);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Network ID NCC
       Command(s): NIDNCC[?]
       Purpose:    Set/Get the NCC network Identification
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Network_ID_NCC(ViSession instrSession, ViInt32 id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(id,0,7) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "NIDNCC %d", id);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Network ID NCC
       Command(s): NIDNCC[?]
       Purpose:    Set/Get the NCC network Identification
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Network_ID_NCC(ViSession instrSession,  ViInt32 _VI_FAR *id)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "NIDNCC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", id);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Off Power
       Command(s): OFFPWR?
       Purpose:    Query the measured results for the previous power measurement at carrier off
       Unit defined by:  AUMT8820_UNIT2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Off_Power(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OFFPWR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Output Level
       Command(s): OLVL[?],SGLVL
       Purpose:    Set/Get the signal generator output level
       Unit defined by:  AUMT8820_OUTPUT_LEVEL_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Output_Level(ViSession instrSession, ViReal64 level, ViInt32 units)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitString[] = {"DBM", "DBU", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(level, -140.00, 113.00) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "OLVL %lf%s", level, unitString[units]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Output Level
       Command(s): OLVL[?],SGLVL
       Purpose:    Set/Get the signal generator output level
       Unit defined by:  AUMT8820_OUTPUT_LEVEL_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Output_Level(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OLVL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RF Output Spectrum Mod Count
       Command(s): ORFSMD_COUNT[?]
       Purpose:    Set/Get the measurement count for the output RF spectrum measurement (modulation)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RF_Output_Spectrum_Mod_Count(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "ORFSMD_COUNT %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Count
       Command(s): ORFSMD_COUNT[?]
       Purpose:    Set/Get the measurement count for the output RF spectrum measurement (modulation)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Count(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ORFSMD_COUNT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RF Output Spectrum Mod Mode
       Command(s): ORFSMD_MEAS[?]
       Purpose:    Set/Get whether to perform the RF output spectrum measurement (modulation)
       Mode defined by:  AUMT8820_RF_OUTPUT_SPECTRUM_MODULATION_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RF_Output_Spectrum_Mod_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "ORFSMD_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Mode
       Command(s): ORFSMD_MEAS[?]
       Purpose:    Set/Get whether to perform the RF output spectrum measurement (modulation)
       Mode defined by:  AUMT8820_RF_OUTPUT_SPECTRUM_MODULATION_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ORFSMD_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RF Output Spectrum Switching Count
       Command(s): ORFSSW_COUNT[?]
       Purpose:    Set/Get the measurement count for the output RF spectrum measurement (switching)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RF_Output_Spectrum_Switching_Count(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "ORFSSW_COUNT %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Count
       Command(s): ORFSSW_COUNT[?]
       Purpose:    Set/Get the measurement count for the output RF spectrum measurement (switching)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Count(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ORFSSW_COUNT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RF Output Spectrum Switching Mode
       Command(s): ORFSSW_MEAS[?]
       Purpose:    Set/Get whether to perform the RF output spectrum measurement (switching)
       Mode defined by:  AUMT8820_RF_OUTPUT_SPECTRUM_SWITCHING_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RF_Output_Spectrum_Switching_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "ORFSSW_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Mode
       Command(s): ORFSSW_MEAS[?]
       Purpose:    Set/Get whether to perform the RF output spectrum measurement (switching)
       Mode defined by:  AUMT8820_RF_OUTPUT_SPECTRUM_SWITCHING_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ORFSSW_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Output Signal Pattern
       Command(s): OSIGPAT[?]
       Purpose:    Set/Get the output signal pattern when call processing is set to off
       Pattern defined by:  AUMT8820_OUTPUT_SIGNAL_PATTERN_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Output_Signal_Pattern(ViSession instrSession, ViInt32 pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"CCH", "TCH", "CCHTCH", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "OSIGPAT %s", commandArray[pattern]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Output Signal Pattern
       Command(s): OSIGPAT[?]
       Purpose:    Set/Get the output signal pattern when call processing is set to off
       Pattern defined by:  AUMT8820_OUTPUT_SIGNAL_PATTERN_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Output_Signal_Pattern(ViSession instrSession, ViInt32  _VI_FAR *pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"CCH", "TCH", "CCHTCH", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OSIGPAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, pattern)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Paging IMSI
       Command(s): PGIMSI[?]
       Purpose:    Set/Get the IMSI type for sending to the phone at paging
       Type defined by:  AUMT8820_PAGING_IMSI_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Paging_IMSI(ViSession instrSession, ViInt32 type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"AUTO", "FIX", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PGIMSI %s", commandArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Paging IMSI
       Command(s): PGIMSI[?]
       Purpose:    Set/Get the IMSI type for sending to the phone at paging
       Type defined by:  AUMT8820_PAGING_IMSI_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Paging_IMSI(ViSession instrSession, ViInt32  _VI_FAR *type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"AUTO", "FIX", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PGIMSI?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, type)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RMS Phase Vector Error
       Command(s): PHASEERR?
       Purpose:    Query the phase vector error for the previous measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RMS_Phase_Vector_Error(ViSession instrSession, ViReal64 _VI_FAR *degree)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", degree);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Phase Vector Error
       Command(s): PPHASEERR?
       Purpose:    Query the peak phase vector error for the previous measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Phase_Vector_Error(ViSession instrSession, ViReal64 _VI_FAR *degree)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PPHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", degree);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Leading
       Command(s): PTLEAD?  time
       Purpose:    Query the power vs time leading
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Leading(ViSession instrSession, ViInt32 time, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(time, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "PTLEAD? %d", time);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Trailing
       Command(s): PTTRAIL?  time
       Purpose:    Query the power vs time trailing
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Trailing(ViSession instrSession, ViInt32 time, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(time, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "PTTRAIL? %d", time);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Measurement Count
       Command(s): PWR_COUNT[?]
       Purpose:    Set/Get the measurement count for the power measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Measurement_Count(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "PWR_COUNT %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Measurement Count
       Command(s): PWR_COUNT[?]
       Purpose:    Set/Get the measurement count for the power measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Measurement_Count(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_COUNT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power VS Time Offset Leading
       Command(s): PWR_LEADTM[?]
       Purpose:    Set/Get the measurement position for power vs time (leading)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_VS_Time_Offset_Leading(ViSession instrSession, ViInt32 time_number, ViReal64 time)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(time_number, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViReal64Range(time, -48.000000, 29.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "PWR_LEADTM %d,%.1f", time_number, time);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Offset Leading
       Command(s): PWR_LEADTM[?]
       Purpose:    Set/Get the measurement position for power vs time (leading)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Offset_Leading(ViSession instrSession, ViInt32 time_number, ViReal64 _VI_FAR *time)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(time_number, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PWR_LEADTM? %d", time_number);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", time);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Measurement Mode
       Command(s): PWR_MEAS[?]
       Purpose:    Set/Get whether to perform power measurements
       Mode defined by:  AUMT8820_POWER_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Measurement_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PWR_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Measurement Mode
       Command(s): PWR_MEAS[?]
       Purpose:    Set/Get whether to perform power measurements
       Mode defined by:  AUMT8820_POWER_MEASUREMENT_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Measurement_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Marker Mode
       Command(s): PWR_MKR[?]
       Purpose:    Set/Get whether to perform marker display on the power measurement view window
       Mode defined by:  AUMT8820_POWER_MARKER_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Marker_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PWR_MKR %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Marker Mode
       Command(s): PWR_MKR[?]
       Purpose:    Set/Get whether to perform marker display on the power measurement view window
       Mode defined by:  AUMT8820_POWER_MARKER_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Marker_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_MKR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Marker Level
       Command(s): PWR_MKRL?
       Purpose:    Query the power at the marker position on the power measurement view window
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Marker_Level(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_MKRL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Marker Position
       Command(s): PWR_MKRP[?]
       Purpose:    Set/Get the marker position on the power measurement view window
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Marker_Position(ViSession instrSession, ViReal64 bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PWR_MKRP %.3lf", bit);

   if (AuMT8820_InvalidViReal64Range(bit, -27.000000, 174.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Marker Position
       Command(s): PWR_MKRP[?]
       Purpose:    Set/Get the marker position on the power measurement view window
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Marker_Position(ViSession instrSession, ViReal64 _VI_FAR *bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_MKRP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", bit);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Off Template Level Unit
       Command(s): PWR_OFFLVL[?]
       Purpose:    Set/Get the unit for off interval level (Line Level 1) in template
       Unit defined by:  AUMT8820_UNIT3_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Off_Template_Level_Unit(ViSession instrSession, ViInt32 unit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "DB", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PWR_OFFLVL %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Off Template Level Unit
       Command(s): PWR_OFFLVL[?]
       Purpose:    Set/Get the unit for off interval level (Line Level 1) in template
       Unit defined by:  AUMT8820_UNIT3_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Off_Template_Level_Unit(ViSession instrSession, ViInt32  _VI_FAR *unit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "DB", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_OFFLVL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, unit)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Storage Method
       Command(s): PWR_STORAGE[?]
       Purpose:    Set/Get the waveform storage method on the power measurement view window
       Method defined by:  AUMT8820_POWER_STORAGE_METHOD_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Storage_Method(ViSession instrSession, ViInt32 method)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"CURR", "AVG", "MAX", "MIN", "FAILED", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PWR_STORAGE %s", commandArray[method]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Storage Method
       Command(s): PWR_STORAGE[?]
       Purpose:    Set/Get thewaveform storage method on the power measurement view window
       Method defined by:  AUMT8820_POWER_STORAGE_METHOD_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Storage_Method(ViSession instrSession, ViInt32  _VI_FAR *method)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"CURR", "AVG", "MAX", "MIN", "FAILED", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_STORAGE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, method)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Template Level
       Command(s): PWR_TEMPLVL[?]
       Purpose:    Set/Get the power template level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Template_Level(ViSession instrSession, ViInt32 line, ViReal64 level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(line, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViReal64Range(level, -110.000000, 10.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   len = sprintf(buf, "PWR_TEMPLVL %d,%.1lf", line, level);

   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Template Level
       Command(s): PWR_TEMPLVL[?]
       Purpose:    Set/Get the power template level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Template_Level(ViSession instrSession, ViInt32 line, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;

   len = sprintf(buf, "PWR_TEMPLVL? %d", line);

   if (AuMT8820_InvalidViInt32Range(line, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Template Measurement Postion
       Command(s): PWR_TEMPPOS[?]
       Purpose:    Set/Get the measurement position in the power template
       Position defined by:  AUMT8820_POWER_TEMPLATE_MEASUREMENT_POSTION_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Template_Measurement_Postion(ViSession instrSession, ViInt32 position)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", "BOTH", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PWR_TEMPPOS %s", commandArray[position]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Template Measurement Postion
       Command(s): PWR_TEMPPOS[?]
       Purpose:    Set/Get the measurement position in the power template
       Position defined by:  AUMT8820_POWER_TEMPLATE_MEASUREMENT_POSTION_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Template_Measurement_Postion(ViSession instrSession, ViInt32  _VI_FAR *position)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", "BOTH", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_TEMPPOS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, position)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Power Template Standard
       Command(s): PWR_TEMPSTD
       Purpose:    Returns the judgement level of each interval in template to default
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Power_Template_Standard(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PWR_TEMPSTD");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power VS Time Trailing
       Command(s): PWR_TRAILTM[?]
       Purpose:    Set/Get the measurement position for power vs. time (trailing)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_VS_Time_Offset_Trailing(ViSession instrSession, ViInt32 time_number, ViReal64 time)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(time_number, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViReal64Range(time, -514.000000, 594.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   len = sprintf(buf, "PWR_TRAILTM %d,%.1lf", time_number, time);

   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Trailing
       Command(s): PWR_TRAILTM[?]
       Purpose:    Set/Get the measurement position for power vs. time (trailing)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Offset_Trailing(ViSession instrSession, ViInt32 time_number, ViReal64 _VI_FAR *time)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(time_number, 1, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PWR_TRAILTM? %d", time_number);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", time);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Measurement Window
       Command(s): PWR_WIN[?]
       Purpose:    Set/Get the waveform display on the power measurement view window
       Window defined by:  AUMT8820_POWER_MEASUREMENT_WINDOW_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Measurement_Window(ViSession instrSession, ViInt32 window)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SLOT", "ONPORT", "LEAD", "TRAIL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PWR_WIN %s", commandArray[window]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Measurement Window
       Command(s): PWR_WIN[?]
       Purpose:    Set/Get the waveform display on the power measurement view window
       Window defined by:  AUMT8820_POWER_MEASUREMENT_WINDOW_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Measurement_Window(ViSession instrSession, ViInt32  _VI_FAR *window)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SLOT", "ONPORT", "LEAD", "TRAIL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_WIN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, window)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Template Result
       Command(s): PWRTEMP?
       Purpose:    Query the template judgement result
       Result defined by:  AUMT8820_RESULT_TEMPLATE_JUDGMENT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Template_Result(ViSession instrSession, ViInt32 _VI_FAR *result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PASS", "FAIL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWRTEMP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, result)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Template Standard
       Command(s): PWR_TEMPSTD
       Purpose:    Set the judgment level of the each interval in template to default
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Template_Standard(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PWR_TEMPSTD");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Ratio
       Command(s): RATIO?
       Purpose:    Query the difference between Tx power and carrier off power
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Ratio(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RATIO?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RF Output Spectrum Resolution Bandwidth
       Command(s): RBWFLT[?]
       Purpose:    Set/Get the RBW for the RF output spectrum measurement (modulation)
       Filter defined by:  AUMT8820_RF_OUTPUT_SPECTRUM_RESOLUTION_BANDWIDTH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RF_Output_Spectrum_Resolution_Bandwidth(ViSession instrSession, ViInt32 filter)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"30KHZ", "100KHZ", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "RBWFLT %s", commandArray[filter]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Resolution Bandwidth
       Command(s): RBWFLT[?]
       Purpose:    Set/Get the RBW for the RF output spectrum measurement (modulation)
       Filter defined by:  AUMT8820_RF_OUTPUT_SPECTRUM_RESOLUTION_BANDWIDTH_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Resolution_Bandwidth(ViSession instrSession, ViInt32  _VI_FAR *filter)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"30KHZ", "100KHZ", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RBWFLT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, filter)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Rx Timeout
       Command(s): RX_TIMEOUT[?]
       Purpose:    Set/Get the maximum measurement time for bit error rate measurement (BER).  A measurement terminates when the measurement time exceeds the set timeout.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Rx_Timeout(ViSession instrSession, ViInt32 time)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(time,1,60) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "RX_TIMEOUT %d", time);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Rx Timeout
       Command(s): RX_TIMEOUT[?]
       Purpose:    Set/Get the maximum measurement time for bit error rate measurement (BER).  A measurement terminates when the measurement time exceeds the set timeout.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Rx_Timeout(ViSession instrSession,  ViInt32 _VI_FAR *time)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RX_TIMEOUT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", time);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Rx Measurement Status
       Command(s): RXMSTAT?
       Purpose:    Query the current reception (Rx) measurement status
       Status defined by:  AUMT8820_MEASUREMENT_STATUS_xxx (some values don't occurr)
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Rx_Measurement_Status(ViSession instrSession, ViInt32 _VI_FAR *status)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString comandArray[] = {"", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RXMSTAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", status);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Active Window
       Command(s): SCRACT[?]
       Purpose:    Set/Get the active window.  The screen to which the window to be activated belongs, must be open
       Window defined by:  AUMT8820_ACTIVE_WINDOW_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Active_Window(ViSession instrSession, ViInt32 window)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SETP", "FMEAS", "MSREP", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SCRACT %s", commandArray[window]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Active Window
       Command(s): SCRACT[?]
       Purpose:    Set/Get the active window.  The screen to which the window to be activated belongs, must be open
       Window defined by:  AUMT8820_ACTIVE_WINDOW_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Active_Window(ViSession instrSession, ViInt32  _VI_FAR *window)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SETP", "FMEAS", "MSREP", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCRACT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /*  Scan(rdBuf, "%d", window);  */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, window)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Screen Select
       Command(s): SCRSEL[?]
       Purpose:    Set/Get the System screen
       Screen defined by:  AUMT8820_SCREEN_SELECT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Screen_Select(ViSession instrSession, ViInt32 screen)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"FMEAS", "PRMSAVE", "PRMRCL", "SYSCFG", "SYSINFO", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SCRSEL %s", commandArray[screen]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Screen Select
       Command(s): SCRSEL[?]
       Purpose:    Set/Get the System screen
       Screen defined by:  AUMT8820_SCREEN_SELECT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Screen_Select(ViSession instrSession, ViInt32  _VI_FAR *screen)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"FMEAS", "PRMSAVE", "PRMRCL", "SYSCFG", "SYSINFO", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCRSEL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, screen)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TCH Slot
       Command(s): SLTNUM[?]
       Purpose:    Set/Get the TCH slot number
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TCH_Slot(ViSession instrSession, ViInt32 slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(slot,2,6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "SLTNUM %d", slot);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TCH Slot
       Command(s): SLTNUM[?]
       Purpose:    Set/Get the TCH slot number
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TCH_Slot(ViSession instrSession,  ViInt32 _VI_FAR *slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SLTNUM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", slot);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set System Combination
       Command(s): SYSCMB[?]
       Purpose:    Set/Get the GSM mode
       System defined by:  AUMT8820_TAG_SELECT_GSM_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_System_Combination(ViSession instrSession, ViInt32 system)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DCS1800", "PCS1900", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SYSCMB %s", commandArray[system]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get System Combination
       Command(s): SYSCMB[?]
       Purpose:    Set/Get the GSM mode
       System defined by:  AUMT8820_TAG_SELECT_GSM_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_System_Combination(ViSession instrSession, ViInt32  _VI_FAR *system)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DCS1800", "PCS1900", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SYSCMB?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, system)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Tag Select
       Command(s): TAGSEL[?]
       Purpose:    Select a screen specific tag to be activated and displayed at the top of the window
       Window defined by: AUMT8820_WINDOW_xxx
       Tag defined by: AUMT8820_TAG_xxx

*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Tag_Select(ViSession instrSession,  ViInt32 tag)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandtagArray[] = {"SETP,COM", "SETP,AUDIO", "SETP,CALL", "SETP,TXMEAS", "SETP,RXMEAS","SETP,FMEAS","FMEAS,PWR", "FMEAS,VSTIME", "FMEAS,TEMP", "FMEAS,MOD", "FMEAS,ORFSMD", "FMEAS,ORFSSW", "FMEAS,BER", "FMEAS,BLER","MSREP,MSREP", "SYSCFG,COMMON", "SYSCFG,PHONE1", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TAGSEL %s",commandtagArray[tag]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Tag Select
       Command(s): TAGSEL[?]
       Purpose:    Select a screen specific tag to be activated and displayed at the top of the window
       Window defined by: AUMT8820_WINDOW_xxx      
       Tag defined by: AUMT8820_TAG_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Tag_Select(ViSession instrSession,  ViInt32 window, ViInt32 _VI_FAR *tag)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandscreenArray[] = {"SETP", "FMEAS", "MSREP", "SYSCFG", 0L};
   static ViString commandtagArray[] = {"COM", "AUDIO", "CALL", "TXMEAS", "RXMEAS","FMEAS","PWR", "VSTIME", "TEMP", "MOD", "ORFSMD", "ORFSSW", "BER", "BLER", "MSREP", "COMMON", "PHONE1", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TAGSEL? %s", commandscreenArray[window]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandtagArray, resBuf0, tag)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Template Mode
       Command(s): TEMP_MEAS[?]
       Purpose:    Set/Get whether to perform the template judgement
       Mode defined by:  AUMT8820_TEMPLATE_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Template_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TEMP_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Template Mode
       Command(s): TEMP_MEAS[?]
       Purpose:    Set/Get whether to perform the template judgement
       Mode defined by:  AUMT8820_TEMPLATE_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Template_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TEMP_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Template Close
       Command(s): TEMPCLS
       Purpose:    Closes the template in the fundamental measurement screen
       Template defined by:  AUMT8820_TEMPLATE_CLOSE_PWR
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Template_Close(ViSession instrSession, ViInt32 template)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PWR", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TEMPCLS %s", commandArray[template]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Template Open
       Command(s): TEMPOPN[?]
       Purpose:    Opens the template in the fundamental measurement screen
       Template defined by:  AUMT8820_TEMPLATE_OPEN_xxx
       Defined in MT8820 WCDMA LabWindows Driver.c
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Template_Open(ViSession instrSession, ViInt32 template)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PWR", "NONE", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TEMPOPN %s", commandArray[template]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Template Open
       Command(s): TEMPOPN[?]
       Purpose:    Opens the template in the fundamental measurement screen
       Template defined by:  AUMT8820_TEMPLATE_OPEN_xxx
       Defined in MT8820 WCDMA LabWindows Driver.c
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Template_Open(ViSession instrSession, ViInt32  _VI_FAR *template)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PWR", "NONE", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TEMPOPN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, template)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TCH Test Pattern
       Command(s): TESTPAT[?]
       Purpose:    Set/Get the TCH test pattern.  Can only be set when call processing is off, or if on, only when status is Idle or Communication
       Pattern defined by:  AUMT8820_TCH_TEST_PATTERN_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TCH_Test_Pattern(ViSession instrSession, ViInt32 pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PN9", "PN15", "ALL0", "ALL1", "ECHO", "VOICE", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TESTPAT %s", commandArray[pattern]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TCH Test Pattern
       Command(s): TESTPAT[?]
       Purpose:    Set/Get the TCH test pattern.  Can only be set when call processing is off, or if on, only when status is Idle or Communication
       Pattern defined by:  AUMT8820_TCH_TEST_PATTERN_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TCH_Test_Pattern(ViSession instrSession, ViInt32  _VI_FAR *pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PN9", "PN15", "ALL0", "ALL1", "ECHO", "VOICE", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TESTPAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, pattern)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TCH Uplink Frequency
       Command(s): TFREQ[?],ULFREQ[?],TXFREQ[?]
       Purpose:    Set/Get the TCH uplink frequency
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TCH_Uplink_Frequency(ViSession instrSession, ViReal64 frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TFREQ %fMHZ", frequency);

   if (AuMT8820_InvalidViReal64Range(frequency, 30.000000, 2700.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TCH Uplink Frequency
       Command(s): TFREQ[?],ULFREQ[?],TXFREQ[?]
       Purpose:    Set/Get the TCH uplink frequency
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TCH_Uplink_Frequency(ViSession instrSession, ViReal64 _VI_FAR *frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   double freqBuf;
   
   len = sprintf(buf, "TFREQ?");

   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", &freqBuf);
   
   *frequency=freqBuf/1000000;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Timing Advance
       Command(s): TMADVANCE[?]
       Purpose:    Set/Get the timing advance
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Timing_Advance(ViSession instrSession, ViInt32 bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(bit,0,63) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "TMADVANCE %d", bit);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Timing Advance
       Command(s): TMADVANCE[?]
       Purpose:    Set/Get the timing advance
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Timing_Advance(ViSession instrSession,  ViInt32 _VI_FAR *bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TMADVANCE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", bit);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Time Alignment
       Command(s): TMALMENT?
       Purpose:    Query the time alignment
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Time_Alignment(ViSession instrSession, ViReal64 _VI_FAR *bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TMALMENT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", bit);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Training Sequence
       Command(s): TSPAT[?]
       Purpose:    Set/Get the training sequence pattern.  Can only be set when call processing is off, or if on, only when status is Idle or Communication
       Pattern defined by:  AUMT8820_TRAINING_SEQUENCE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Training_Sequence(ViSession instrSession, ViInt32 pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"TSC0", "TSC1", "TSC2", "TSC3", "TSC4", "TSC5", "TSC6", "TSC7", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TSPAT %s", commandArray[pattern]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Training Sequence
       Command(s): TSPAT[?]
       Purpose:    Set/Get the training sequence pattern.  Can only be set when call processing is off, or if on, only when status is Idle or Communication
       Pattern defined by:  AUMT8820_TRAINING_SEQUENCE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Training_Sequence(ViSession instrSession, ViInt32  _VI_FAR *pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"TSC0", "TSC1", "TSC2", "TSC3", "TSC4", "TSC5", "TSC6", "TSC7", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TSPAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, pattern)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Total
       Command(s): TTL_BER?
       Purpose:    Query the bit error rate, and the number of error bits, received bits and sample bits in the bit error rate measurement
       Type defined by: AUMT8820_TYPE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Total(ViSession instrSession, ViInt32 type, ViInt32 *judgment,
																						  ViReal64 _VI_FAR *rate,
                                                                                          ViReal64 _VI_FAR *error_bits,
                                                                                          ViReal64 _VI_FAR *rx_bits,
                                                                                          ViReal64 _VI_FAR *sample_bits)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString typeArray[] = {"FER", "CIB", "CII", "FAST", "BER", "GPRSBER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_BER? %s", typeArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f,%f", judgment, rate, error_bits, rx_bits, sample_bits);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Error Total
       Command(s): TTL_CARRFERR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of frequency error measurements
       Unit defined by:  AUMT8820_UNIT_xxx
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Error_Total(ViSession instrSession, ViInt32 unit, ViInt32 *judgment, ViReal64 _VI_FAR *avg_freq, ViReal64 _VI_FAR *max_freq, ViReal64 _VI_FAR *min_freq)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"HZ", "PPM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_CARRFERR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f", judgment, avg_freq, max_freq, min_freq);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Lower Total
       Command(s): TTL_LMODPWR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the RF output spectrum (modulation, lower)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Lower_Total(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *avg_level, ViReal64 _VI_FAR *max_level, ViReal64 _VI_FAR *min_level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_LMODPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f,%f,%f", avg_level, max_level, min_level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Lower Total
       Command(s): TTL_LSWPWR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the RF output spectrum (switching, lower)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Lower_Total(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *avg_level, ViReal64 _VI_FAR *max_level, ViReal64 _VI_FAR *min_level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_LSWPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f,%f,%f", avg_level, max_level, min_level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Magnitude Error Total
       Command(s): TTL_MAGTDERR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the magnitude error measurements
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Magnitude_Error_Total(ViSession instrSession, ViInt32 *judgment, ViReal64 _VI_FAR *avg_percent, ViReal64 _VI_FAR *max_percent, ViReal64 _VI_FAR *min_percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_MAGTDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f", judgment, avg_percent, max_percent, min_percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Flatness MAX Power Total
       Command(s): TTL_MAXPWR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the difference between Tx power and burst on interval maximum power
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Flatness_MAX_Power_Total(ViSession instrSession, ViInt32 *judgment, ViReal64 _VI_FAR *avg_level, ViReal64 _VI_FAR *max_level, ViReal64 _VI_FAR *min_level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_MAXPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f", judgment, avg_level, max_level, min_level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Flatness MIN Power Total
       Command(s): TTL_MINPWR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the difference between Tx power and burst on interval minimum power
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Flatness_MIN_Power_Total(ViSession instrSession, ViInt32 *judgment, ViReal64 _VI_FAR *avg_level, ViReal64 _VI_FAR *max_level, ViReal64 _VI_FAR *min_level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_MINPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f", judgment, avg_level, max_level, min_level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Off Power Total
       Command(s): TTL_OFFPWR?  unit
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of power at carrier off
       Unit defined by:  AUMT8820_UNIT2_xxx
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Off_Power_Total(ViSession instrSession, ViInt32 unit, ViInt32 *judgment, ViReal64 _VI_FAR *avg_level, ViReal64 _VI_FAR *max_level, ViReal64 _VI_FAR *min_level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_OFFPWR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f", judgment, avg_level, max_level, min_level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RMS Phase Vector Error Total
       Command(s): TTL_PHASEERR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the phase vector error measurements
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RMS_Phase_Vector_Error_Total(ViSession instrSession, ViInt32 *judgment, ViReal64 _VI_FAR *avg_deg, ViReal64 _VI_FAR *max_deg, ViReal64 _VI_FAR *min_deg)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_PHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f", judgment, avg_deg, max_deg, min_deg);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get PEAK Phase Vector Error Total
       Command(s): TTL_PPHASEERR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the peak phase vector error measurements
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_PEAK_Phase_Vector_Error_Total(ViSession instrSession, ViInt32 *judgment, ViReal64 _VI_FAR *avg_deg, ViReal64 _VI_FAR *max_deg, ViReal64 _VI_FAR *min_deg)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_PPHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f", judgment, avg_deg, max_deg, min_deg);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Leading Total
       Command(s): TTL_PTLEAD?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of power vs time (leading)
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Leading_Total(ViSession instrSession, ViInt32 *judgment1, ViReal64 _VI_FAR *avg_level1, ViReal64 _VI_FAR *max_level1, ViReal64 _VI_FAR *min_level1,
                                                                                   ViInt32 *judgment2, ViReal64 _VI_FAR *avg_level2, ViReal64 _VI_FAR *max_level2, ViReal64 _VI_FAR *min_level2,
                                                                                   ViInt32 *judgment3, ViReal64 _VI_FAR *avg_level3, ViReal64 _VI_FAR *max_level3, ViReal64 _VI_FAR *min_level3,
                                                                                   ViInt32 *judgment4, ViReal64 _VI_FAR *avg_level4, ViReal64 _VI_FAR *max_level4, ViReal64 _VI_FAR *min_level4,
                                                                                   ViInt32 *judgment5, ViReal64 _VI_FAR *avg_level5, ViReal64 _VI_FAR *max_level5, ViReal64 _VI_FAR *min_level5,
                                                                                   ViInt32 *judgment6, ViReal64 _VI_FAR *avg_level6, ViReal64 _VI_FAR *max_level6, ViReal64 _VI_FAR *min_level6)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_PTLEAD?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f,%d,%f,%f,%f,%d,%f,%f,%f,%d,%f,%f,%f,%d,%f,%f,%f,%d,%f,%f,%f",
     judgment1, avg_level1, max_level1, min_level1,
     judgment2, avg_level2, max_level2, min_level2,
     judgment3, avg_level3, max_level3, min_level3,
     judgment4, avg_level4, max_level4, min_level4,
     judgment5, avg_level5, max_level5, min_level5,
     judgment6, avg_level6, max_level6, min_level6);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Trailing Total
       Command(s): TTL_PTTRAIL?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of power vs time (trailing)
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Trailing_Total(ViSession instrSession, ViInt32 *judgment1, ViReal64 _VI_FAR *avg_level1, ViReal64 _VI_FAR *max_level1, ViReal64 _VI_FAR *min_level1,
                                                                                    ViInt32 *judgment2, ViReal64 _VI_FAR *avg_level2, ViReal64 _VI_FAR *max_level2, ViReal64 _VI_FAR *min_level2,
                                                                                    ViInt32 *judgment3, ViReal64 _VI_FAR *avg_level3, ViReal64 _VI_FAR *max_level3, ViReal64 _VI_FAR *min_level3,
                                                                                    ViInt32 *judgment4, ViReal64 _VI_FAR *avg_level4, ViReal64 _VI_FAR *max_level4, ViReal64 _VI_FAR *min_level4,
                                                                                    ViInt32 *judgment5, ViReal64 _VI_FAR *avg_level5, ViReal64 _VI_FAR *max_level5, ViReal64 _VI_FAR *min_level5,
                                                                                    ViInt32 *judgment6, ViReal64 _VI_FAR *avg_level6, ViReal64 _VI_FAR *max_level6, ViReal64 _VI_FAR *min_level6)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_PTTRAIL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f,%d,%f,%f,%f,%d,%f,%f,%f,%d,%f,%f,%f,%d,%f,%f,%f,%d,%f,%f,%f",
     judgment1, avg_level1, max_level1, min_level1,
     judgment2, avg_level2, max_level2, min_level2,
     judgment3, avg_level3, max_level3, min_level3,
     judgment4, avg_level4, max_level4, min_level4,
     judgment5, avg_level5, max_level5, min_level5,
     judgment6, avg_level6, max_level6, min_level6);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Template Judgement Total
       Command(s): TTL_PWRTEMP?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of a template
       Result defined by:  AUMT8820_RESULT_TEMPLATE_JUDGMENT_xxx
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Template_Judgement_Total(ViSession instrSession, ViInt32 *judgment, ViInt32 _VI_FAR *avg_result, ViInt32 _VI_FAR *max_result, ViInt32 _VI_FAR *min_result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString resultArray[] = {"PASS", "UNUSED", "UNUSED", "UNUSED", "FAIL", "UNUSED", "UNUSED", "UNUSED", "UNUSED", "NOT_JUDGED", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViChar    resBuf1[BUFFER_SIZE];
   ViChar    resBuf2[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_PWRTEMP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d,%d,%d", judgment, avg_result, max_result, min_result);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Ratio Total
       Command(s): TTL_RATIO?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the difference between Tx power and carrier off power
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Ratio_Total(ViSession instrSession, ViInt32 *judgment, ViReal64 _VI_FAR *avg_level, ViReal64 _VI_FAR *max_level, ViReal64 _VI_FAR *min_level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_RATIO?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f", judgment, avg_level, max_level, min_level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Time Alignment Total
       Command(s): TTL_TMALMENT?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the time alignment
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Time_Alignment_Total(ViSession instrSession, ViInt32 *judgment, ViReal64 _VI_FAR *avg_bit, ViReal64 _VI_FAR *max_bit, ViReal64 _VI_FAR *min_bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_TMALMENT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f", judgment, avg_bit, max_bit, min_bit);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Tx Power Total
       Command(s): TTL_TXPWR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the transmission (Tx) power measurement for the phone
       Unit defined by:  AUMT8820_UNIT2_xxx
       Judgment defined by:   AUMT8820_TEMPLATE_JUDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TX_Power_Total(ViSession instrSession, ViInt32 unit, ViInt32 *judgment, ViReal64 _VI_FAR *avg_level, ViReal64 _VI_FAR *max_level, ViReal64 _VI_FAR *min_level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_TXPWR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%f,%f,%f", judgment,  avg_level, max_level, min_level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Upper Total
       Command(s): TTL_UMODPWR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the RF output spectrum (modulation, upper)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Upper_Total(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *avg_level, ViReal64 _VI_FAR *max_level, ViReal64 _VI_FAR *min_level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_UMODPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f,%f,%f", avg_level, max_level, min_level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Upper Total
       Command(s): TTL_USWPWR?
       Purpose:    Query the average, maximum and minimum values among measured results for the average count of the RF output spectrum (switching, upper)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Upper_Total(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *avg_level, ViReal64 _VI_FAR *max_level, ViReal64 _VI_FAR *min_level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_USWPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f,%f,%f", avg_level, max_level, min_level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Tx Measurement Status
       Command(s): TXMSTAT?
       Purpose:    Query the current transmission (Tx) measurement status
       Status defined by:  AUMT8820_MEASUREMENT_STATUS_xxx (some values not used)
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Tx_Measurement_Status(ViSession instrSession, ViInt32 _VI_FAR *status)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString comandArray[] = {"", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TXMSTAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", status);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Tx Power
       Command(s): TXPWR?
       Purpose:    Query the measured result of the transmission (Tx) power for the previously measured phone
       Unit defined by:  AUMT8820_UNIT2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Tx_Power(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TXPWR? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Tx Power Level
       Command(s): TXPWRLVL?
       Purpose:    Query the monitor value of the transmission (Tx) power
       Unit defined by:  AUMT8820_UNIT2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Tx_Power_Level(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TXPWRLVL? %s", unitArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Mod Upper
       Command(s): UMODPWR?
       Purpose:    Query the RF output spectrum (modulation, upper)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Mod_Upper(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "UMODPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Output Spectrum Switching Upper
       Command(s): USWPWR?
       Purpose:    Query the RF output spectrum (switching, upper)
       Offset_freq defined by:  AUMT8820_OFFSET_FREQUENCY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Output_Spectrum_Switching_Upper(ViSession instrSession, ViInt32 offset_freq, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString offset_freqArray[] = {"OF0", "OF100", "OF200", "OF250", "OF400", "OF600", "OF800", "OF1000", "OF1200", "OF1400", "OF1600", "OF1800", "OF2000", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "USWPWR? %s", offset_freqArray[offset_freq]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Voltage Display
       Command(s): VDSPL[?]
       Purpose:    Set/Get the power voltage display unit when the RF output (SG) setting unit is dBu
       Display defined by:  AUMT8820_VOLTAGE_DISPLAY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Voltage_Display(ViSession instrSession, ViInt32 display)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"EMF", "TERM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "VDSPL %s", commandArray[display]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Voltage Display
       Command(s): VDSPL[?]
       Purpose:    Set/Get the power voltage display unit when the RF output (SG) setting unit is dBu
       Display defined by:  AUMT8820_VOLTAGE_DISPLAY_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Voltage_Display(ViSession instrSession, ViInt32  _VI_FAR *display)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"EMF", "TERM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "VDSPL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, display)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power VS Time Mode
       Command(s): VSTIME_MEAS[?]
       Purpose:    Set/Get whether to measure power vs. time or not
       Mode defined by:  AUMT8820_POWER_VS_TIME_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_VS_Time_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "VSTIME_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power VS Time Mode
       Command(s): VSTIME_MEAS[?]
       Purpose:    Set/Get whether to measure power vs. time or not
       Mode defined by:  AUMT8820_POWER_VS_TIME_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_VS_Time_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "VSTIME_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Close View Window
       Command(s): VWCLS
       Purpose:    Closes the view window
       View defined by: AUMT8820_VIEW_WINDOW_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Close_View_Window(ViSession instrSession, ViInt32 view)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString comandArray[] = {"PWR", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "VWCLS %s", comandArray[view]);

   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Open View Window
       Command(s): VWOPN[?]
       Purpose:    Set/Get the type of view window that is currently open
       Mode defined by:  AUMT8820_VIEW_WINDOW_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Open_View_Window(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PWR", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "VWOPN %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Open View Window
       Command(s): VWOPN[?]
       Purpose:    Set/Get the type of view window that is currently open
       Mode defined by:  AUMT8820_VIEW_WINDOW_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Open_View_Window(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PWR", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "VWOPN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Window Size
       Command(s): WINSIZE[?]
       Purpose:    Set/Get the size of the window
       Size defined by:  AUMT8820_WINDOW_SIZE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Window_Size(ViSession instrSession, ViInt32 size)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"100", "70", "50", "30", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "WINSIZE %s", commandArray[size]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Window Size
       Command(s): WINSIZE[?]
       Purpose:    Set/Get the size of the window
       Size defined by:  AUMT8820_WINDOW_SIZE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Window_Size(ViSession instrSession, ViInt32  _VI_FAR *size)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"100", "70", "50", "30", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "WINSIZE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, size)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Wave Memory IQ
       Command(s): XMC[?]
       Purpose:    Reads, processes, etc. the waveform data on the IQ signal
       Iq defined by:  AUMT8820_PHASE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Wave_Memory_IQ(ViSession instrSession, ViInt32 iq, ViInt32 pos, ViInt32 n, ViChar _VI_FAR data[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString iqArray[] = {"0", "1", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[LARGE_BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(pos, 0, 1608) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(n, 0, 1609) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "XMC? %s,%d,%d", iqArray[iq], pos, n);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, LARGE_BUFFER_M1, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s" ,data);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Wave Memory Tx Power
       Command(s): XMD[?]
       Purpose:    Reads, processes, etc. the waveform data on the power measurement
       Iq defined by:  AUMT8820_PHASE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Wave_Memory_Tx_Power(ViSession instrSession, ViInt32 pos, ViInt32 n, ViChar _VI_FAR data[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[LARGE_BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(pos, 0, 1608) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(n, 0, 1609) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "XMD? %d,%d", pos, n);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, LARGE_BUFFER_M1, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s" ,data);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Wave Memory Magnitude Error
       Command(s): XMN[?]
       Purpose:    Reads, processes, etc. the waveform data on the magnitude error
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Wave_Memory_Magnitude_Error(ViSession instrSession, ViInt32 pos, ViInt32 n, ViChar _VI_FAR data[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[LARGE_BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(pos, 0, 1608) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(n, 0, 1609) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "XMN? %d,%d", pos, n);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, LARGE_BUFFER_M1, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s" ,data);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Wave Memory Phase Error
       Command(s): XMP?
       Purpose:    Reads, processes, etc. the waveform data on the phase error
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Wave_Memory_Phase_Error(ViSession instrSession, ViInt32 pos, ViInt32 n, ViChar _VI_FAR data[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[LARGE_BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(pos, 0, 1608) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(n, 0, 1609) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "XMP? %d,%d", pos, n);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, LARGE_BUFFER_M1, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s" ,data);

   return auMT8820_status;

}

/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/
